const express = require('express');
const fs = require('fs');
const path = require('path');
const app = express();

const DATA_FILE = path.join(__dirname, 'games-data.json');

// Загрузка данных
let gamesData = loadData();
const locks = {}; // Блокировки для каждого gameId

// Инициализация блокировок
gamesData.forEach(game => {
  locks[game.gameId] = false;
});

// --- Функции работы с данными ---
function loadData() {
    const data = fs.readFileSync(DATA_FILE, 'utf8');
    console.log('Данные загружены из файла');
    return JSON.parse(data);
}

function saveData() {
  fs.writeFileSync(DATA_FILE, JSON.stringify(gamesData, null, 2));
  console.log('Данные сохранены');
}

// --- Безопасные операции ---
async function safeModify(gameId, operation) {
  // Ожидаем разблокировки
  while (locks[gameId]) {
    await new Promise(resolve => setTimeout(resolve, 10));
  }
  
  locks[gameId] = true;
  const game = gamesData.find(g => g.gameId === gameId);
  let result;

  if (game) {
    if (operation === 'increment') game.currentValue += 1;
    if (operation === 'decrement') game.currentValue -= 1;
    if (operation === 'autoIncrement') game.currentValue += game.incrementValue;
    result = { success: true, newValue: game.currentValue };
  } else {
    result = { success: false, error: "Game not found" };
  }

  locks[gameId] = false;
  return result;
}

// --- Автоматические процессы ---
// Автоинкремент с блокировкой
async function autoUpdate() {
  console.log('--- Запуск автоинкремента ---');
  for (const game of gamesData) {
    await safeModify(game.gameId, 'autoIncrement');
    console.log(`[Auto] ${game.gameId}: ${game.currentValue}`);
  }
}
setInterval(autoUpdate, 20 * 60 * 1000); // Каждые 20 минут

// Автосохранение
setInterval(saveData, 60 * 60 * 1000); // Каждые час

// --- API ---
app.get('/api/games', (req, res) => {
  res.json(gamesData);
});

app.post('/api/games/:gameId/increment', async (req, res) => {
  try {
    const result = await safeModify(req.params.gameId, 'increment');
    res.json(result);
  } catch (e) {
    res.status(500).json({ error: "Server error" });
  }
});

app.post('/api/games/:gameId/decrement', async (req, res) => {
  try {
    const result = await safeModify(req.params.gameId, 'decrement');
    res.json(result);
  } catch (e) {
    res.status(500).json({ error: "Server error" });
  }
});

// --- Graceful shutdown ---
process.on('SIGINT', () => {
  console.log('\nСервер выключается...');
  saveData();
  process.exit();
});

// Запуск
const PORT = 3000;
app.listen(PORT, () => {
  console.log(`Сервер запущен на http://localhost:${PORT}`);
  console.log(`Автосохранение каждые 60 минут`);
  console.log(`Автоинкремент каждые 20 минут`);
});